bl_info = {
    "name": "PLC-Lab 3D Studio Tools",
    "author": "MHJ-Software GmbH & Co. KG",
    "version": (2, 6), # Version erhöht für Bugfix
    "blender": (2, 80, 0),
    "location": "View3D > Sidebar > PLC-Lab 3D",
    "description": "Tools to prepare models for export. Refactored for easy skill extension.",
    "warning": "The 'Delete All Materials' and 'Delete All Collections' functions are irreversible.",
    "doc_url": "",
    "category": "Export",
}

import bpy
import re

# --- HELPER FUNCTIONS ---
def get_all_children_recursively(obj):
    children = []
    for child in obj.children:
        children.append(child)
        children.extend(get_all_children_recursively(child))
    return children

# (Die Abschnitte SKILL-DEFINITIONEN, OPERATOR-BASISKLASSEN und DYNAMISCHE OPERATOR-ERSTELLUNG bleiben unverändert)
# =============================================================================
# === ZENTRALISIERTE SKILL-DEFINITIONEN =======================================
# =============================================================================
SKILL_SENSORS = [
    ("lightbarrier", "{LightBarrier}", "Skill for a light barrier"),
    ("limitswitch", "{LimitSwitch}", "Skill for a limit switch"),
    ("switchorbutton", "{SwitchOrButton}", "Skill for a generic switch or button"),
    ("ultrasonicsensor", "{UltrasonicSensor}", "Skill for an ultrasonic sensor"),
    ("weightsensor", "{WeightSensor}", "Skill for a weight sensor"),
]
SKILL_ACTUATORS = [
    ("conveyor", "{Conveyor}", "Skill for a conveyor belt"),
    ("drive_2_point", "{Drive-2-Point}", "Skill for a two-point drive"),
    ("drive_2_point_rotation", "{Drive-2-Point-Rotation}", "Skill for a two-point rotational drive"),
    ("drive_2_point_physic", "{Drive-2-Point-Physic}", "Skill for a physics-based two-point drive"),
    ("drivelink", "{DriveLink}", "Skill to link drives"),
    ("endlessrotation", "{EndlessRotation}", "Skill for continuous rotation"),
    ("exactposition", "{ExactPosition}", "Skill for precise positioning"),
    ("joint", "{Joint}", "Skill for a physical joint"),
    ("lamp", "{Lamp}", "Skill for a simple lamp"),
    ("linkunlink_player", "{LinkUnlinkPlayer}", "Skill to link or unlink player objects"),
    ("liquidcontainer", "{LiquidContainer}", "Skill for a container holding liquids"),
    ("magnet", "{Magnet}", "Skill for a magnet"),
    ("robotkinematics", "{RobotKinematics}", "Skill for robot kinematics"),
    ("rotateonmovement", "{RotateOnMovement}", "Skill for rotation based on movement"),
    ("servodrive", "{ServoDrive}", "Skill for a servo drive"),
    ("servodrive_rotation", "{ServoDrive-Rotation}", "Skill for a rotational servo drive"),
    ("slidingpot", "{SlidingPot}", "Skill for a sliding potentiometer"),
]
SKILL_OTHER = [
    ("label_3d", "{3D-Label}", "Skill for a 3D label"),
    ("collider", "{Collider}", "Skill to define a collider"),
    ("creator", "{Creator}", "Skill for an object creator"),
    ("destroyer", "{Destroyer}", "Skill for an object destroyer"),
    ("hide", "{Hide}", "Skill to hide an object"),
    ("moveablelight", "{MoveableLight}", "Skill for a moveable light source"),
    ("objecthelper", "{ObjectHelper}", "Skill for an object helper"),
    ("playaudio", "{PlayAudio}", "Skill to play an audio file"),
    ("player", "{Player}", "Skill to define the player object"),
    ("scenelight", "{SceneLight}", "Skill for a static scene light"),
    ("tools", "{Tools}", "Skill for tools"),
]
SKILL_MODIFIERS_LIGHTS = [
    ("white", "{WHITE}", "Set indication light color to white"),
    ("green", "{GREEN}", "Set indication light color to green"),
    ("red", "{RED}", "Set indication light color to red"),
    ("gray", "{GRAY}", "Set indication light color to gray"),
    ("blue", "{BLUE}", "Set indication light color to blue"),
    ("yellow", "{YELLOW}", "Set indication light color to yellow"),
]
SKILL_MODIFIERS_BUTTONS = [
    ("nc", "{NC}", "Normally Closed"),
    ("emergency", "{emergency}", "Switch is an Emergency Stop"),
    ("main", "{main}", "Main Switch"),
    ("rotate", "{rotate}", "Toggle Switch"),
    ("switch", "{switch}", "Button/Switch"),
]
SKILL_MODIFIERS_AXIS = [
    ("x", "{X}", "Reference X axis"), ("y", "{Y}", "Reference Y axis"), ("z", "{Z}", "Reference Z axis"),
    ("neg_x", "{-X}", "Reference negative X axis"), ("neg_y", "{-Y}", "Reference negative Y axis"), ("neg_z", "{-Z}", "Reference negative Z axis"),
]
SKILL_MODIFIERS_OTHER = [
    ("mesh", "{mesh}", "Use a mesh collider"),
    ("inductive", "{inductive}", "Change LimitSwitch behavior to inductive"),
    ("capacitive", "{capacitive}", "Change LimitSwitch behavior to capacitive"),
    ("optical", "{optical}", "Change LimitSwitch behavior to optical"),
	

]
# =============================================================================
# === OPERATOR-BASISKLASSEN UND ERSTELLUNG ====================================
# =============================================================================
class PLCLAB_OT_add_skill_base(bpy.types.Operator):
    bl_label = "Add Skill Base"
    bl_options = {'REGISTER', 'UNDO'}
    skill_string: bpy.props.StringProperty()
    @classmethod
    def poll(cls, context): return context.selected_objects
    def execute(self, context): raise NotImplementedError()
class PLCLAB_OT_add_standard_skill(PLCLAB_OT_add_skill_base):
    bl_idname = "plclab.add_standard_skill"
    bl_label = "Add Standard Skill"
    def execute(self, context):
        count = 0
        for obj in context.selected_objects:
            if self.skill_string not in obj.name:
                obj.name += self.skill_string
                count += 1
        self.report({'INFO'}, f"Added skill '{self.skill_string}' to {count} objects.")
        return {'FINISHED'}
COLOR_TAGS_TO_REMOVE = [item[1] for item in SKILL_MODIFIERS_LIGHTS]
COLOR_REGEX = re.compile("|".join(re.escape(tag) for tag in COLOR_TAGS_TO_REMOVE))
class PLCLAB_OT_add_color_skill(PLCLAB_OT_add_skill_base):
    bl_idname = "plclab.add_color_skill"
    bl_label = "Add Color Skill"
    def execute(self, context):
        count = 0
        for obj in context.selected_objects:
            name_without_colors = COLOR_REGEX.sub("", obj.name)
            new_name = name_without_colors + self.skill_string
            if obj.name != new_name:
                obj.name = new_name
                count += 1
        self.report({'INFO'}, f"Set color skill '{self.skill_string}' on {count} objects.")
        return {'FINISHED'}
dynamically_created_classes = []
def make_invoke(skill_str):
    def invoke(self, context, event):
        self.skill_string = skill_str
        return self.execute(context)
    return invoke
def create_skill_operators(skill_definitions, base_class):
    for ident, skill, desc in skill_definitions:
        op_idname = f"plclab.add_{ident}_skill"
        op_classname = f"PLCLAB_OT_add_{ident}_skill"
        class_props = {
            "bl_idname": op_idname,
            "bl_label": skill,
            "bl_description": desc,
            "invoke": make_invoke(skill)
        }
        new_class = type(op_classname, (base_class,), class_props)
        dynamically_created_classes.append(new_class)
create_skill_operators(SKILL_SENSORS, PLCLAB_OT_add_standard_skill)
create_skill_operators(SKILL_ACTUATORS, PLCLAB_OT_add_standard_skill)
create_skill_operators(SKILL_OTHER, PLCLAB_OT_add_standard_skill)
create_skill_operators(SKILL_MODIFIERS_BUTTONS, PLCLAB_OT_add_standard_skill)
create_skill_operators(SKILL_MODIFIERS_AXIS, PLCLAB_OT_add_standard_skill)
create_skill_operators(SKILL_MODIFIERS_OTHER, PLCLAB_OT_add_standard_skill)
create_skill_operators(SKILL_MODIFIERS_LIGHTS, PLCLAB_OT_add_color_skill)

# =============================================================================
# === STATISCHE OPERATOREN ====================================================
# =============================================================================
class PLCLAB_OT_apply_scale_and_export_fbx(bpy.types.Operator):
    bl_idname = "plclab.apply_scale_and_export_fbx"
    bl_label = "Apply Scale & Export as FBX"
    filepath: bpy.props.StringProperty(subtype='FILE_PATH')
    def execute(self, context):
        # ... (Code unverändert) ...
        original_active = context.view_layer.objects.active
        original_selection = context.selected_objects[:]
        if bpy.ops.object.select_all.poll(): bpy.ops.object.select_all(action='DESELECT')
        for obj in context.scene.objects:
            if obj.type in {'MESH', 'EMPTY', 'CURVE', 'SURFACE', 'META', 'FONT', 'ARMATURE', 'LATTICE', 'LIGHT', 'CAMERA'}:
                obj.select_set(True)
        if context.selected_objects: bpy.ops.object.transform_apply(scale=True, location=False, rotation=False)
        if bpy.ops.object.select_all.poll(): bpy.ops.object.select_all(action='DESELECT')
        for obj in original_selection: obj.select_set(True)
        context.view_layer.objects.active = original_active
        if not self.filepath.lower().endswith(".fbx"): self.filepath += ".fbx"
        bpy.ops.export_scene.fbx(filepath=self.filepath, apply_scale_options='FBX_SCALE_UNITS', use_selection=False, object_types={'ARMATURE', 'CAMERA', 'EMPTY', 'LIGHT', 'MESH', 'OTHER'}, use_mesh_modifiers=True, bake_anim=False)
        self.report({'INFO'}, f"Scene successfully exported to: {self.filepath}")
        return {'FINISHED'}
    def invoke(self, context, event):
        blend_file_name = bpy.path.basename(context.blend_data.filepath)
        self.filepath = blend_file_name.replace(".blend", ".fbx") if blend_file_name else "export.fbx"
        context.window_manager.fileselect_add(self)
        return {'RUNNING_MODAL'}

class PLCLAB_OT_select_all_children(bpy.types.Operator):
    bl_idname = "plclab.select_all_children"
    bl_label = "Select All Children"
    @classmethod
    def poll(cls, context): return context.selected_objects
    def execute(self, context):
        to_select = set(context.selected_objects)
        for obj in context.selected_objects: to_select.update(get_all_children_recursively(obj))
        bpy.ops.object.select_all(action='DESELECT')
        for obj in to_select: obj.select_set(True)
        self.report({'INFO'}, f"Selected {len(to_select)} objects (parents + children).")
        return {'FINISHED'}

class PLCLAB_OT_select_skill_objects(bpy.types.Operator):
    bl_idname = "plclab.select_skill_objects"
    bl_label = "Select All Skill Objects"
    @classmethod
    def poll(cls, context): return bool(context.view_layer.objects)
    def execute(self, context):
        bpy.ops.object.select_all(action='DESELECT')
        skill_pattern = re.compile(r"\{[^}]*\}")
        selected_objects = []
        for obj in context.view_layer.objects:
            if skill_pattern.search(obj.name):
                obj.select_set(True)
                selected_objects.append(obj)
        if selected_objects: context.view_layer.objects.active = selected_objects[0]
        self.report({'INFO'}, f"Selected {len(selected_objects)} skill objects.")
        return {'FINISHED'}

class PLCLAB_OT_select_non_skill_objects(bpy.types.Operator):
    bl_idname = "plclab.select_non_skill_objects"
    bl_label = "Select All Non-Skill Objects"
    @classmethod
    def poll(cls, context): return bool(context.view_layer.objects)
    def execute(self, context):
        bpy.ops.object.select_all(action='DESELECT')
        skill_pattern = re.compile(r"\{[^}]*\}")
        selected_objects = []
        for obj in context.view_layer.objects:
            if not skill_pattern.search(obj.name):
                obj.select_set(True)
                selected_objects.append(obj)
        if selected_objects: context.view_layer.objects.active = selected_objects[0]
        self.report({'INFO'}, f"Selected {len(selected_objects)} non-skill objects.")
        return {'FINISHED'}

# ### START: KORRIGIERTER OPERATOR ###
class PLCLAB_OT_make_all_visible(bpy.types.Operator):
    """Unhides all objects in the viewport, including those disabled with the monitor icon"""
    bl_idname = "plclab.make_all_visible"
    bl_label = "Make All Objects Visible"
    
    @classmethod
    def poll(cls, context):
        return bool(bpy.data.objects)
        
    def execute(self, context):
        bpy.ops.object.hide_view_clear()        
        return {'FINISHED'}
# ### ENDE: KORRIGIERTER OPERATOR ###

class PLCLAB_OT_select_all_empties(bpy.types.Operator):
    bl_idname = "plclab.select_all_empties"
    bl_label = "Select All Empties"
    @classmethod
    def poll(cls, context): return bool(context.view_layer.objects)
    def execute(self, context):
        bpy.ops.object.select_all(action='DESELECT')
        empties = [obj for obj in context.view_layer.objects if obj.type == 'EMPTY']
        for obj in empties: obj.select_set(True)
        if empties: context.view_layer.objects.active = empties[0]
        self.report({'INFO'}, f"{len(empties)} Empties selected.")
        return {'FINISHED'}
        
class PLCLAB_OT_duplicate_with_children(bpy.types.Operator):
    bl_idname = "plclab.duplicate_with_children"
    bl_label = "Duplicate Sel. Objects with Children"
    @classmethod
    def poll(cls, context): return context.selected_objects
    def execute(self, context):
        objects_to_duplicate = set(context.selected_objects)
        for obj in context.selected_objects: objects_to_duplicate.update(get_all_children_recursively(obj))
        bpy.ops.object.select_all(action='DESELECT')
        for obj in objects_to_duplicate: obj.select_set(True)
        bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked": False, "mode": 'TRANSLATION'}, TRANSFORM_OT_translate={"value": (0.01, 0, 0)})
        self.report({'INFO'}, f"Duplicated {len(objects_to_duplicate)} objects with children.")
        return {'FINISHED'}
        
class PLCLAB_OT_remove_skill_tags(bpy.types.Operator):
    bl_idname = "plclab.remove_skill_tags"
    bl_label = "Remove Skills"
    @classmethod
    def poll(cls, context): return context.selected_objects
    def execute(self, context):
        count = 0
        for obj in context.selected_objects:
            new_name = re.sub(r"\{[^}]*\}", "", obj.name)
            if new_name != obj.name:
                obj.name = new_name
                count += 1
        self.report({'INFO'}, f"Removed skill tags from {count} objects.")
        return {'FINISHED'}
# ... (Restliche Operatoren für Cleanup bleiben unverändert) ...
class PLCLAB_OT_remove_selected_materials(bpy.types.Operator):
    bl_idname = "plclab.remove_selected_materials"
    bl_label = "Remove Materials on Selected"
    @classmethod
    def poll(cls, context): return context.selected_objects
    def execute(self, context):
        count = 0
        for obj in context.selected_objects:
            if obj.type == 'MESH' and obj.material_slots:
                obj.data.materials.clear()
                count += 1
        self.report({'INFO'}, f"Removed materials from {count} selected objects.")
        return {'FINISHED'}
class PLCLAB_OT_delete_all_materials(bpy.types.Operator):
    bl_idname = "plclab.delete_all_materials"
    bl_label = "Delete All Materials"
    @classmethod
    def poll(cls, context): return len(bpy.data.materials) > 0
    def execute(self, context):
        mats = list(bpy.data.materials)
        num_deleted = len(mats)
        for m in mats: bpy.data.materials.remove(m, do_unlink=True)
        self.report({'INFO'}, f"{num_deleted} materials were deleted.")
        return {'FINISHED'}
    def invoke(self, context, event): return context.window_manager.invoke_confirm(self, event)
class PLCLAB_OT_delete_none_pl3s_materials(bpy.types.Operator):
    bl_idname = "plclab.delete_none_pl3s_materials"
    bl_label = "Delete Non-PLC-Lab 3D Materials"
    @classmethod
    def poll(cls, context): return len(bpy.data.materials) > 0
    def execute(self, context):
        count = 0
        for mat in list(bpy.data.materials):
            if not mat.name.startswith("pl3s_"):
                bpy.data.materials.remove(mat, do_unlink=True)
                count += 1
        self.report({'INFO'}, f"Deleted {count} non-PLC-Lab 3D materials.")
        return {'FINISHED'}
class PLCLAB_OT_remove_duplicate_materials(bpy.types.Operator):
    bl_idname = "plclab.remove_duplicate_materials"
    bl_label = "Remove Duplicate Materials"
    @classmethod
    def poll(cls, context): return len(bpy.data.materials) > 0
    def execute(self, context):
        pattern = re.compile(r"(.*)\.\d{3}$")
        replaced, removed = 0, 0
        for mat in list(bpy.data.materials):
            m = pattern.match(mat.name)
            if not m: continue
            base = bpy.data.materials.get(m.group(1))
            if not base: continue
            for obj in bpy.data.objects:
                for slot in obj.material_slots:
                    if slot.material == mat:
                        slot.material = base
                        replaced += 1
            bpy.data.materials.remove(mat, do_unlink=True)
            removed += 1
        self.report({'INFO'}, f"Removed {removed} duplicates and replaced {replaced} slots.")
        return {'FINISHED'}
class PLCLAB_OT_delete_all_collections(bpy.types.Operator):
    bl_idname = "plclab.delete_all_collections"
    bl_label = "Dissolve Specific Collections"
    bl_description = "Resolves collections with '{do_not_render_in_studio}' objects. Content is moved to the parent."
    @classmethod
    def poll(cls, context): return len(context.scene.collection.children) > 0
    def execute(self, context):
        def get_all_objects_recursively(collection):
            objects = list(collection.objects)
            for child_collection in collection.children: objects.extend(get_all_objects_recursively(child_collection))
            return objects
        collections_to_dissolve = []
        search_string = "{do_not_render_in_studio}"
        for coll in list(context.scene.collection.children):
            if any(search_string in obj.name for obj in get_all_objects_recursively(coll)):
                collections_to_dissolve.append(coll)
        if not collections_to_dissolve:
            self.report({'INFO'}, "No matching collections to dissolve found.")
            return {'CANCELLED'}
        dissolved_count = 0
        for coll in collections_to_dissolve:
            parent_collection = next((p for p in bpy.data.collections if coll.name in p.children), context.scene.collection)
            for obj in list(coll.objects): parent_collection.objects.link(obj)
            for child_coll in list(coll.children): parent_collection.children.link(child_coll)
            try:
                bpy.data.collections.remove(coll)
                dissolved_count += 1
            except Exception as e:
                self.report({'WARNING'}, f"Could not delete collection '{coll.name}': {e}")
        self.report({'INFO'}, f"{dissolved_count} collection(s) dissolved. Content was preserved.")
        return {'FINISHED'}
    def invoke(self, context, event): return context.window_manager.invoke_confirm(self, event)
    
# =============================================================================
# === UI PANELS ===============================================================
# =============================================================================
# (Der UI-Teil bleibt unverändert, da die bl_idname des Buttons gleich ist)
class PLCLAB_PT_main_panel(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'PLC-Lab 3D'
    bl_label = "PLC-Lab 3D Tools"
    def draw(self, context):
        layout = self.layout
        v = bl_info["version"]
        layout.label(text=f"Version {v[0]}.{v[1]}", icon='PLUGIN')

class PLCLAB_PT_SkillPanelBase(PLCLAB_PT_main_panel):
    bl_parent_id = "PLCLAB_PT_main_panel"
    bl_options = {'DEFAULT_CLOSED'}
    skill_definitions = [] 
    def draw(self, context):
        layout = self.layout
        layout.operator("plclab.remove_skill_tags", icon='X')
        for ident, _, _ in self.skill_definitions:
            layout.operator(f"plclab.add_{ident}_skill")
class PLCLAB_PT_skill_sensors(PLCLAB_PT_SkillPanelBase):
    bl_label = "Skill Sensors"
    skill_definitions = SKILL_SENSORS
class PLCLAB_PT_skill_actuators(PLCLAB_PT_SkillPanelBase):
    bl_label = "Skill Actuators"
    skill_definitions = SKILL_ACTUATORS
class PLCLAB_PT_skill_other(PLCLAB_PT_SkillPanelBase):
    bl_label = "Skill Other"
    skill_definitions = SKILL_OTHER

class PLCLAB_PT_tools(PLCLAB_PT_main_panel):
    bl_parent_id = "PLCLAB_PT_main_panel"
    bl_label = "Tools"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        
        layout.operator("plclab.select_all_children", icon='OUTLINER_OB_GROUP_INSTANCE')
        layout.operator("plclab.select_skill_objects", icon='VIEWZOOM')
        layout.operator("plclab.select_non_skill_objects", icon='SNAP_OFF')
        layout.operator("plclab.duplicate_with_children", icon='DUPLICATE')
        layout.operator("plclab.make_all_visible", icon='HIDE_OFF')
        layout.operator("plclab.select_all_empties", icon='EMPTY_DATA')

class PLCLAB_PT_skill_modifiers(PLCLAB_PT_main_panel):
    bl_parent_id = "PLCLAB_PT_main_panel"
    bl_label = "Skill Modifiers"
    bl_options = {'DEFAULT_CLOSED'}
    def draw(self, context): pass
class PLCLAB_PT_skill_modifiers_indication_lights(PLCLAB_PT_skill_modifiers):
    bl_parent_id = "PLCLAB_PT_skill_modifiers"
    bl_label = "For Indication Lights"
    bl_options = {'DEFAULT_CLOSED'}
    def draw(self, context):
        layout = self.layout
        layout.operator("plclab.remove_skill_tags", icon='X')
        row = layout.row(align=True)
        row.operator("plclab.add_white_skill")
        row.operator("plclab.add_green_skill")
        row = layout.row(align=True)
        row.operator("plclab.add_red_skill")
        row.operator("plclab.add_gray_skill")
        row = layout.row(align=True)
        row.operator("plclab.add_blue_skill")
        row.operator("plclab.add_yellow_skill")
class PLCLAB_PT_skill_modifiers_buttons(PLCLAB_PT_SkillPanelBase):
    bl_parent_id = "PLCLAB_PT_skill_modifiers"
    bl_label = "For Buttons / Switch"
    skill_definitions = SKILL_MODIFIERS_BUTTONS
class PLCLAB_PT_skill_modifiers_other(PLCLAB_PT_SkillPanelBase):
    bl_parent_id = "PLCLAB_PT_skill_modifiers"
    bl_label = "Other"
    skill_definitions = SKILL_MODIFIERS_OTHER
class PLCLAB_PT_skill_modifiers_reference_axis(PLCLAB_PT_skill_modifiers):
    bl_parent_id = "PLCLAB_PT_skill_modifiers"
    bl_label = "Reference Axis"
    bl_options = {'DEFAULT_CLOSED'}
    def draw(self, context):
        layout = self.layout
        layout.operator("plclab.remove_skill_tags", icon='X')
        row = layout.row(align=True)
        row.operator("plclab.add_x_skill")
        row.operator("plclab.add_y_skill")
        row.operator("plclab.add_z_skill")
        row = layout.row(align=True)
        row.operator("plclab.add_neg_x_skill")
        row.operator("plclab.add_neg_y_skill")
        row.operator("plclab.add_neg_z_skill")
class PLCLAB_PT_cleanup(PLCLAB_PT_main_panel):
    bl_parent_id = "PLCLAB_PT_main_panel"
    bl_label = "Cleanup"
    bl_options = {'DEFAULT_CLOSED'}
    def draw(self, context):
        layout = self.layout
        layout.operator("plclab.remove_selected_materials")
        layout.operator("plclab.remove_duplicate_materials")
        layout.operator("plclab.delete_none_pl3s_materials")
        layout.operator("plclab.delete_all_materials")
        layout.operator("plclab.delete_all_collections", icon='OUTLINER_COLLECTION')
class PLCLAB_PT_export(PLCLAB_PT_main_panel):
    bl_parent_id = "PLCLAB_PT_main_panel"
    bl_label = "Export"
    bl_options = {'DEFAULT_CLOSED'}
    def draw(self, context):
        self.layout.operator("plclab.apply_scale_and_export_fbx")
        
# =============================================================================
# === REGISTRIERUNG ===========================================================
# =============================================================================
# (Auch die Registrierung bleibt unverändert)
static_classes = (
    PLCLAB_OT_apply_scale_and_export_fbx,
    PLCLAB_OT_select_all_children,
    PLCLAB_OT_select_skill_objects,
    PLCLAB_OT_select_non_skill_objects,
    PLCLAB_OT_make_all_visible,
    PLCLAB_OT_select_all_empties,
    PLCLAB_OT_duplicate_with_children,
    PLCLAB_OT_remove_skill_tags,
    PLCLAB_OT_remove_selected_materials,
    PLCLAB_OT_remove_duplicate_materials,
    PLCLAB_OT_delete_none_pl3s_materials,
    PLCLAB_OT_delete_all_materials,
    PLCLAB_OT_delete_all_collections,
    PLCLAB_PT_main_panel,
    PLCLAB_PT_tools,
    PLCLAB_PT_skill_sensors,
    PLCLAB_PT_skill_actuators,
    PLCLAB_PT_skill_other,
    PLCLAB_PT_skill_modifiers,
    PLCLAB_PT_skill_modifiers_indication_lights,
    PLCLAB_PT_skill_modifiers_buttons,
    PLCLAB_PT_skill_modifiers_reference_axis,
    PLCLAB_PT_skill_modifiers_other,
    PLCLAB_PT_cleanup,
    PLCLAB_PT_export,
)

classes_to_register = static_classes + tuple(dynamically_created_classes)

def register():
    for cls in classes_to_register:
        bpy.utils.register_class(cls)

def unregister():
    for cls in reversed(classes_to_register):
        bpy.utils.unregister_class(cls)

if __name__ == "__main__":
    register()